#include "mpi.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "SpaceShape.h"
#include "meshamgadget.h"
#include <mesham.h>
#define BITSPERDIM 20
#define OPTIMISEFOREXECUTION 1
/*
	For optimiseforexecution set to 1 if want to optimise for execution (cost of memory), 0 otherwise.
	There is a much longer runtime if set to 0, but will save some memory (as this does not keep indexed (reference) record of the ordered key)
	Suggest always set to 1
*/ 
MESHPeanoHandle addcurve(char *,char *);
void buildcurve(char *);
int comparePHKeys(const void *a,const void *b);
char * getVarname(MESHPeanoHandle phandle);

MESHPeanoHandle peanohandle=0;
typedef struct PHkeynode
{	
	long long key;			
	ObjectItem * shapeitem;
	int lastupdate;
	struct PHkeynode * next;
	struct PHkeynode * prev;
} PHkey;

typedef struct PHcurvenode
{	
	char * variablename;
	MESHPeanoHandle mypeanohandle;	
	double conversionfactor;
	ShapeSpace * shape;
	int lastupdated;	
	PHkey * keys; 
	PHkey ** sortedkeys;
	struct PHcurvenode * next;
	struct PHcurvenode * prev;
} PHcurve;
void freecurvekeys(PHcurve *);
long long computePHkey(int, int, int,PHcurve *);
PHcurve * getcurvenode(char *);
PHcurve * phhead=NULL;

long long MESHGADgetPHCurvetotalcells(MESHPeanoHandle phandle)
{
	// dont care about handle for now!
	return  (((long long)1)<<(3*BITSPERDIM));
}

MESHPeanoHandle MESHGADregisterPHCurve(char * varname,MESHShapeSpaceHandle shandle)
{
	char * space=getShapeVarname(shandle);
	return addcurve(varname,space);
}

long long MESHGADgetPHCurveelementkey(MESHPeanoHandle phandle,int index)
{
	char * varname = getVarname(phandle);
	buildcurve(varname);	// build curve if not already done (and ensure up to date)
	PHcurve * crv=getcurvenode(varname);
	if (OPTIMISEFOREXECUTION == 1)
	{	
		return crv->sortedkeys[index]->key;
	} else {
		int i;
		PHkey * currenthead=crv->keys;	
		for (i=0;i<index;i++)
			if (currenthead != NULL) currenthead=currenthead->next;
		if (currenthead==NULL) return;
		return currenthead->key;
	}
}

char * getVarname(MESHPeanoHandle phandle)
{
	PHcurve * ch=phhead;
	//printf("%p\n",ch->next);
	while (ch != NULL)
	{		
		if (ch->mypeanohandle == phandle) return ch->variablename;		
		ch=ch -> next;		
	}	
}

int MESHGADgetPHCurveelementorderadded(MESHPeanoHandle phandle,int index)
{
	char * varname = getVarname(phandle);
	buildcurve(varname);	// build curve if not already done (and ensure up to date)
	PHcurve * crv=getcurvenode(varname);
	if (OPTIMISEFOREXECUTION == 1)
	{	
		return crv->sortedkeys[index]->shapeitem->orderadded;
	} else {
		int i;
		PHkey * currenthead=crv->keys;
		for (i=0;i<index;i++)
			if (currenthead != NULL) currenthead=currenthead->next;
		if (currenthead==NULL) return;
		return currenthead->shapeitem->orderadded;
	}
}

void * MESHGADgetPHCurveelement(MESHPeanoHandle phandle,int index)
{
	char * varname = getVarname(phandle);
	buildcurve(varname);	// build curve if not already done (and ensure up to date)
	PHcurve * crv=getcurvenode(varname);
	if (OPTIMISEFOREXECUTION == 1)
	{	
		return crv->sortedkeys[index]->shapeitem->data;
	} else {
		int i;
		PHkey * currenthead=crv->keys;
		for (i=0;i<index;i++)
			if (currenthead != NULL) currenthead=currenthead->next;
		if (currenthead==NULL) return;
		return currenthead->shapeitem->data;
	}
}

void freecurvekeys(PHcurve * crv)
{
	PHkey* currenthead=crv->keys;
	while (currenthead != NULL)
	{
		PHkey * oldh=currenthead;
		currenthead=currenthead->next;
		free(oldh);
	}
}

void buildcurve(char * varname)
{
	PHcurve * crv=getcurvenode(varname);
	if (crv->shape->masterupdatecode == crv->lastupdated) return;	// all up to date, no need to rebuild curve	
	freecurvekeys(crv);	// free all the PH keys in the current curve as will rebuild them all 
	if (OPTIMISEFOREXECUTION == 1 && crv->sortedkeys != NULL) free(crv->sortedkeys);	// free sorted keys as will rebuild (if unempty)
	crv->keys=NULL;
	ObjectItem * topitem=crv->shape->contents;
	if (topitem==NULL) return;
	int points=0;
	while (topitem != NULL)
	{
		PHkey * newkey=malloc(sizeof(PHkey));
		newkey->shapeitem=topitem;
		newkey->key=computePHkey(topitem->coords[0],topitem->coords[1],topitem->coords[2],crv);
		newkey->lastupdate=crv->shape->masterupdatecode;
		newkey->next=crv->keys;	
		newkey->prev=NULL;
		if (crv->keys!=NULL)
		{
			crv->keys->prev=newkey;
		}
		crv->keys=newkey;	
		topitem=topitem->next;
		points++;
	}	
	crv->lastupdated=crv->shape->masterupdatecode;
	/*
		Now Sort the list
	*/
	//PHkey * kys[points];
	PHkey * sortedkeys[points];	
	PHkey * top=crv->keys;
	int j=0;
	while (top != NULL)
	{
		sortedkeys[j]=top;
		j++;
		top=top->next;
	}
	qsort(sortedkeys,points,sizeof(PHkey *),comparePHKeys);
	// below will rebuild the DL list using the order obtained by quicksort
	for (j=0;j<points;j++)
	{
		if (j < points - 1)
		{
			sortedkeys[j]->next=sortedkeys[j + 1];
		} else {
			sortedkeys[j]->next=NULL;
		}
		if (j > 0) 
		{
			sortedkeys[j]->prev=sortedkeys[j - 1];
		} else {
			sortedkeys[j]->prev=NULL;
		}		
	}
	crv->keys=sortedkeys[0];	// point to first key in the list
	
	if (OPTIMISEFOREXECUTION == 1) 
	{
		crv->sortedkeys=malloc(sizeof(PHkey *) * points);
		memcpy(crv->sortedkeys,sortedkeys,sizeof(PHkey *) * points);
	} 		
}

int comparePHKeys(const void *a,const void *b)
{
	if (   (*((PHkey **) a))->key < (*((PHkey **) b))->key ) return -1;
	if (   (*((PHkey **) a))->key > (*((PHkey **) b))->key ) return 1;
	return 0;
}

static int quadrants[24][2][2][2] = {
  /* rotx=0, roty=0-3 */
  {{{0, 7}, {1, 6}}, {{3, 4}, {2, 5}}},
  {{{7, 4}, {6, 5}}, {{0, 3}, {1, 2}}},
  {{{4, 3}, {5, 2}}, {{7, 0}, {6, 1}}},
  {{{3, 0}, {2, 1}}, {{4, 7}, {5, 6}}},
  /* rotx=1, roty=0-3 */
  {{{1, 0}, {6, 7}}, {{2, 3}, {5, 4}}},
  {{{0, 3}, {7, 4}}, {{1, 2}, {6, 5}}},
  {{{3, 2}, {4, 5}}, {{0, 1}, {7, 6}}},
  {{{2, 1}, {5, 6}}, {{3, 0}, {4, 7}}},
  /* rotx=2, roty=0-3 */
  {{{6, 1}, {7, 0}}, {{5, 2}, {4, 3}}},
  {{{1, 2}, {0, 3}}, {{6, 5}, {7, 4}}},
  {{{2, 5}, {3, 4}}, {{1, 6}, {0, 7}}},
  {{{5, 6}, {4, 7}}, {{2, 1}, {3, 0}}},
  /* rotx=3, roty=0-3 */
  {{{7, 6}, {0, 1}}, {{4, 5}, {3, 2}}},
  {{{6, 5}, {1, 2}}, {{7, 4}, {0, 3}}},
  {{{5, 4}, {2, 3}}, {{6, 7}, {1, 0}}},
  {{{4, 7}, {3, 0}}, {{5, 6}, {2, 1}}},
  /* rotx=4, roty=0-3 */
  {{{6, 7}, {5, 4}}, {{1, 0}, {2, 3}}},
  {{{7, 0}, {4, 3}}, {{6, 1}, {5, 2}}},
  {{{0, 1}, {3, 2}}, {{7, 6}, {4, 5}}},
  {{{1, 6}, {2, 5}}, {{0, 7}, {3, 4}}},
  /* rotx=5, roty=0-3 */
  {{{2, 3}, {1, 0}}, {{5, 4}, {6, 7}}},
  {{{3, 4}, {0, 7}}, {{2, 5}, {1, 6}}},
  {{{4, 5}, {7, 6}}, {{3, 2}, {0, 1}}},
  {{{5, 2}, {6, 1}}, {{4, 3}, {7, 0}}}
};


static int rotxmap_table[24] = { 4, 5, 6, 7, 8, 9, 10, 11,
  12, 13, 14, 15, 0, 1, 2, 3, 17, 18, 19, 16, 23, 20, 21, 22
};

static int rotymap_table[24] = { 1, 2, 3, 0, 16, 17, 18, 19,
  11, 8, 9, 10, 22, 23, 20, 21, 14, 15, 12, 13, 4, 5, 6, 7
};

static int rotx_table[8] = { 3, 0, 0, 2, 2, 0, 0, 1 };
static int roty_table[8] = { 0, 1, 1, 2, 2, 3, 3, 0 };

static int sense_table[8] = { -1, -1, -1, +1, +1, -1, -1, -1 };

long long computePHkey(int x, int y, int z, PHcurve * crv)	// taken directly from Gadget 2 code
{
	x -= crv->shape->dimcorner[0];
	y -= crv->shape->dimcorner[1];
	z -= crv->shape->dimcorner[2];
	x *= crv->conversionfactor;
	y *= crv->conversionfactor;
	z *= crv->conversionfactor;
	int i, quad, bitx, bity, bitz;
  	int mask, rotation, rotx, roty, sense;
  	long long key;

 	mask = 1 << (BITSPERDIM - 1);
  	key = 0;
  	rotation = 0;
  	sense = 1;
  	
  	for(i = 0; i < BITSPERDIM; i++, mask >>= 1)
    {
      bitx = (x & mask) ? 1 : 0;
      bity = (y & mask) ? 1 : 0;
      bitz = (z & mask) ? 1 : 0;

      quad = quadrants[rotation][bitx][bity][bitz];

      key <<= 3;
      key += (sense == 1) ? (quad) : (7 - quad);

      rotx = rotx_table[quad];
      roty = roty_table[quad];
      sense *= sense_table[quad];

     while(rotx > 0)
     {
	  		rotation = rotxmap_table[rotation];
	  		rotx--;
	  }

      while(roty > 0)
      {
	  		rotation = rotymap_table[rotation];
	  		roty--;
	  	}
    }
  return key;
}

int addcurve(char * varname,char * space)
{
	PHcurve * newcrv=malloc(sizeof(PHcurve));
	ShapeSpace * theshape=getSpaceShapenode(space);
	newcrv->variablename=malloc(sizeof(char) * strlen(varname) + 2);
	strcpy(newcrv->variablename,varname);	
	newcrv->shape=theshape;
	newcrv->sortedkeys=NULL;
	newcrv->keys=NULL;
	newcrv->mypeanohandle=peanohandle;
	peanohandle++;
	newcrv->lastupdated=-1;
	newcrv->conversionfactor = 1.0 / theshape->len * (((long long) 1) << BITSPERDIM);
	newcrv->next=phhead;	
	newcrv->prev=NULL;
	if (phhead!=NULL)
	{
		phhead->prev=newcrv;
	}
	phhead=newcrv;	
	return newcrv->mypeanohandle;
}

void MESHGADfreePHCurve(MESHPeanoHandle phandle)
{
	char * varname = getVarname(phandle);
	PHcurve * ch=getcurvenode(varname);
	if (ch==NULL) return;
	if ((ch->prev) != NULL)
	{
		(ch->prev)->next=ch->next;		
	} else {		
		phhead=ch->next;
	}
	if (ch->next !=NULL) {(ch->next)->prev=ch->prev;}
	if (ch->variablename != NULL) {free(ch->variablename);}		
	free(ch);
}

PHcurve * getcurvenode(char * varname)
{
	PHcurve * ch=phhead;
	//printf("%p\n",ch->next);
	while (ch != NULL)
	{		
		if (strcmp(ch->variablename,varname) == 0) return ch;		
		ch=ch -> next;		
	}	
	return NULL;
}
